import React, { useState, useContext } from "react";
import axios from "axios";
import { AuthContext } from "../contexts/AuthContext";

export default function AddCar() {
  const { token } = useContext(AuthContext);

  const [formData, setFormData] = useState({
    name: "",
    model: "",
    color: "",
    type: "",
    price: "",
    fuelType: "",
    image: null,
  });
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState("");
  const [success, setSuccess] = useState("");

  const handleChange = (e) => setFormData({ ...formData, [e.target.name]: e.target.value });
  const handleFileChange = (e) => setFormData({ ...formData, image: e.target.files[0] });

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);
    setError("");
    setSuccess("");

    try {
      const data = new FormData();
      Object.entries(formData).forEach(([key, value]) => { if (value) data.append(key, value); });

      await axios.post("http://localhost:8000/api/admin/cars", data, {
        headers: {
          Authorization: `Bearer ${token}`,
          "Content-Type": "multipart/form-data",
        },
      });

      setSuccess("Car added successfully!");
      setFormData({
        name: "",
        model: "",
        color: "",
        type: "",
        price: "",
        fuelType: "",
        image: null,
      });
    } catch (err) {
      console.error(err);
      if (err.response?.status === 401) setError("Unauthorized. Login as admin.");
      else setError("Failed to add car. Try again.");
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="max-w-xl mx-auto bg-white shadow-lg rounded-2xl p-8 mt-6">
      <h2 className="text-2xl font-semibold mb-6">Add New Car</h2>
      {error && <p className="text-red-600 mb-4">{error}</p>}
      {success && <p className="text-green-600 mb-4">{success}</p>}
      <form onSubmit={handleSubmit} className="space-y-4">
        <input type="text" name="name" value={formData.name} onChange={handleChange} placeholder="Car Name" required className="w-full border p-2 rounded-md" />
        <input type="text" name="model" value={formData.model} onChange={handleChange} placeholder="Model" required className="w-full border p-2 rounded-md" />
        <input type="text" name="color" value={formData.color} onChange={handleChange} placeholder="Color" required className="w-full border p-2 rounded-md" />
        <select name="type" value={formData.type} onChange={handleChange} required className="w-full border p-2 rounded-md">
          <option value="">Select Type</option>
          <option value="SUV">SUV</option>
          <option value="Sedan">Sedan</option>
          <option value="Hatchback">Hatchback</option>
          <option value="Truck">Truck</option>
          <option value="Coupe">Coupe</option>
        </select>
        <input type="number" name="price" value={formData.price} onChange={handleChange} placeholder="Price" required className="w-full border p-2 rounded-md" />
        <select name="fuelType" value={formData.fuelType} onChange={handleChange} required className="w-full border p-2 rounded-md">
          <option value="">Select Fuel Type</option>
          <option value="Diesel">Diesel</option>
          <option value="Petrol">Petrol</option>
          <option value="Electric">Electric</option>
          <option value="Hybrid">Hybrid</option>
        </select>
        <input type="file" accept="image/*" onChange={handleFileChange} className="w-full border p-2 rounded-md" />
        <button type="submit" disabled={loading} className="w-full bg-blue-600 text-white py-2 rounded-md hover:bg-blue-700">{loading ? "Saving..." : "Add Car"}</button>
      </form>
    </div>
  );
}
